/*---------------------------------------------------------------------------*\

	FILE....: DIGITS.CPP
	TYPE....: C++ Module
	AUTHOR..: David Rowe
	DATE....: 28/7/98

	Digit collection module for VPB API.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include "apifunc.h"
#include "digbuf.h"
#include "timer.h"
#include "wobbly.h"
#include "generic.h"
#include <assert.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

/*---------------------------------------------------------------------------*\

								DEFINES

\*--------------------------------------------------------------------------*/

#define	MAX_TERM			31	// maximum number of terminate digits
#define	DEF_INTER_DIGIT_TIME_OUT	5000	// default
#define	DEF_DIGIT_TIME_OUT		60000	// default
#define	DEF_MAX_DIGITS			1		
#define	MAX_DIGITS			31	// max size of digit buffer
#define	NUM_VALID_DIG			16	// number of valid digits
#define	SLEEPMS				20	// sleep time

// states for each channels digit collection

#define	INACTIVE					0		
#define	ACTIVE						1

/*---------------------------------------------------------------------------*\

								TYPEDEFS

\*--------------------------------------------------------------------------*/

typedef struct {
	char	term_digits[MAX_TERM+1];// string of digits to terminate collection
	USHORT	max_digits;		// terminate after this many digits collected
	ULONG	digit_time_out;		// max total time for digit collection (ms)
	ULONG	inter_digit_time_out;	// max time between digits (ms)
	USHORT	so_far;			// number of digits so far
	Timer	*digit_timer;		// total timer
	Timer	*inter_digit_timer;	// inter digit timer
	int		state;	
	char	*buf;			// ptr to desination buffer;			
	USHORT	size;			// size of destination buffer;
	short	voxtron;		// check_buffer_voxtron() called if voxtron is enabled				
	short	record;			//
	int		async;		// true if async get digits, 0 otherwise
} VPB_DIGITS_INT;

/*---------------------------------------------------------------------------*\

								STATICS

\*--------------------------------------------------------------------------*/

static VPB_DIGITS_INT *digits;		// ptr to state array
static void *dbuf, *dbufrec;		// digit buffers
// prevents MMQ interfering with other threads
static GENERIC_CRITICAL_SECTION	DigitsSect;

static int NumCh;

static char valid_dig[] = "0123456789*#ABCD";
static char *term_str[] = {
	"Terminating Digit",
	"Maximum Digits",
	"Time Out",
	"Inter-Digit Time Out",
	"Destination Buffer Full",
	"Invalid VPB_DIGIT event"
};

/*---------------------------------------------------------------------------*\

						LOCAL FUNCTION HEADERS

\*--------------------------------------------------------------------------*/

static int digit_match(char dig, char *term_digits);
static void	validate_digits(char *digits);
void check_buffer(int handle);
void check_buffer_voxtron(int handle);
void check_buffer_record(int handle);
 
/*---------------------------------------------------------------------------*\

								FUNCTIONS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: digits_open
	AUTHOR..: David Rowe
	DATE....: 27/7/98

	Initialises the digits module.

\*--------------------------------------------------------------------------*/

void digits_open(USHORT numch) {
	digits = new VPB_DIGITS_INT[numch];
	assert(digits != NULL);

	int i;
	for(i=0; i<numch; i++) {
		digits[i].term_digits[0]= 0;
		digits[i].max_digits = DEF_MAX_DIGITS;
		digits[i].digit_time_out = DEF_DIGIT_TIME_OUT;
		digits[i].inter_digit_time_out = DEF_INTER_DIGIT_TIME_OUT;
		digits[i].so_far = 0;
		digits[i].state = INACTIVE;
		digits[i].async = 0;
		digits[i].voxtron = 0;
		digits[i].record = 0;
		digits[i].digit_timer = new Timer;
		digits[i].inter_digit_timer = new Timer;
	}

	dbuf = digbuf_open(numch, MAX_DIGITS);
	dbufrec = digbuf_open(numch, MAX_DIGITS);
   	GenericInitializeCriticalSection(&DigitsSect);
	NumCh = numch;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: digits_close
	AUTHOR..: David Rowe
	DATE....: 27/7/98

	Closes the digits module.

\*--------------------------------------------------------------------------*/

void digits_close() {
	int i;
	for(i=0; i<NumCh; i++) {
		delete digits[i].digit_timer;
		delete digits[i].inter_digit_timer;
	}

	delete [] digits;
	digbuf_close(dbuf);
	digbuf_close(dbufrec);
	GenericDeleteCriticalSection(&DigitsSect);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_get_digits_async
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Gets a buffer of digits based on the conditions passed to the function.
	Returns immediately, and posts a comletetion event when the buffer
	has been obtained.

	If this function is called while already collecting digits (by a
	prvious call to this function), the digit collection is reset, 
	and digits collected so far are discarded.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_get_digits_async(int handle, VPB_DIGITS *newdig, char *buf)
//	int		handle		handle for this channel
//	VPB_DIGITS	*newdig		get digits parameters
//	char		*buf		destination buffer
{
	VPB_DIGITS_INT	*d;
	
	try {
		ValidHandleCheck(handle);

		// dont want MMQ adding digits at same time we are reading
		// them out.

		GenericEnterCriticalSection(&DigitsSect);

		// copy params to local storage

		d = &digits[handle];		
		
		validate_digits(newdig->term_digits);
		strcpy(d->term_digits,newdig->term_digits);
		if (newdig->max_digits > MAX_DIGITS)
			throw Wobbly(VPBAPI_DIGITS_MAX_DIGITS_TOO_BIG);
		
		d->max_digits = newdig->max_digits;
		d->digit_time_out = newdig->digit_time_out;
		d->inter_digit_time_out = newdig->inter_digit_time_out;
		d->so_far = 0;
		d->state = ACTIVE;
		d->buf = buf;
		d->voxtron = 0;
		d->digit_timer->timer_start();
		d->inter_digit_timer->timer_stop();	// timer in reset
		d->async = 1;
 		check_buffer(handle);

		GenericLeaveCriticalSection(&DigitsSect);

		// start by reading digbuf for this channel

	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_get_digits_async"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: get_digits_async
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Gets a buffer of digits based on the conditions passed to the function.
	Returns immediately, and posts a comletetion event when the buffer
	has been obtained.

	If this function is called while already collecting digits (by a
	prvious call to this function), the digit collection is reset, 
	and digits collected so far are discarded.
	Similar to vpb_get_digits_async except it may terminate if the user digit 
	buffer (destination buffer) is full.

\*--------------------------------------------------------------------------*/

int get_digits_async(int handle, VPB_DIGITS *newdig, char *buf, unsigned short size)
//	int			handle		handle for this channel
//	VPB_DIGITS	*newdig		get digits parameters
//	char		*buf		destination buffer
// unsigned short size		size of destination buffer
{
	VPB_DIGITS_INT	*d;
	
	try {
		ValidHandleCheck(handle);

		// dont want MMQ adding digits at same time we are reading
		// them out.

		GenericEnterCriticalSection(&DigitsSect);

		// copy params to local storage

		d = &digits[handle];		
		
		validate_digits(newdig->term_digits);
		strcpy(d->term_digits,newdig->term_digits);
		if (newdig->max_digits > MAX_DIGITS)
			throw Wobbly(VPBAPI_DIGITS_MAX_DIGITS_TOO_BIG);
		
		d->max_digits = newdig->max_digits;
		d->digit_time_out = newdig->digit_time_out;
		d->inter_digit_time_out = newdig->inter_digit_time_out;
		d->so_far = 0;
		d->state = ACTIVE;
		d->buf = buf;
		d->voxtron = 1;
		d->size = size;
		d->digit_timer->timer_start();
		d->inter_digit_timer->timer_stop();	//timer in reset
		d->async = 1;
 		check_buffer_voxtron(handle);

		GenericLeaveCriticalSection(&DigitsSect);

		// start by reading digbuf for this channel

	}

	catch(Wobbly w){
		return(RunTimeError(w,"get_digits_async"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: get_digits_record_async
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Gets a buffer of digits based on the conditions passed to the function.
	Returns immediately, and posts a completion event when the buffer
	has been obtained.

	If this function is called while already collecting digits (by a
	prvious call to this function), the digit collection is reset, 
	and digits collected so far are discarded.
	Similar to vpb_get_digits_async except it is used by the MAL_recordXXXX().
	It copies the digits to a secondary buffer "dbufrec".

\*--------------------------------------------------------------------------*/

int get_digits_record_async(int handle, VPB_DIGITS *newdig, char *buf)
//	int			handle		handle for this channel
//	VPB_DIGITS	*newdig		get digits parameters
//	char		*buf		destination buffer
{
	VPB_DIGITS_INT	*d;
	
	try {
		ValidHandleCheck(handle);

		// dont want MMQ adding digits at same time we are reading
		// them out.

		GenericEnterCriticalSection(&DigitsSect);

		// copy params to local storage

		d = &digits[handle];		
		
		validate_digits(newdig->term_digits);
		strcpy(d->term_digits,newdig->term_digits);
		if (newdig->max_digits > MAX_DIGITS)
			throw Wobbly(VPBAPI_DIGITS_MAX_DIGITS_TOO_BIG);
		
		d->max_digits = newdig->max_digits;
		d->digit_time_out = newdig->digit_time_out;
		d->so_far = 0;
		d->state = ACTIVE;
		d->buf = buf;
		d->record = 1;
		d->digit_timer->timer_start();
		d->async = 1;
 		check_buffer_record(handle);

		GenericLeaveCriticalSection(&DigitsSect);

		// start by reading digbuf for this channel

	}

	catch(Wobbly w){
		return(RunTimeError(w,"get_digits_record_async"));
	}
	
	return(VPB_OK);
}
/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_get_digits_sync
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Gets a buffer of digits based on the conditions passed to the function.
	Returns when digit buffer has been obtained, with appropriate
	VPB_DIGIT_XXX return code.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_get_digits_sync(int handle, VPB_DIGITS *newdig, char *buf)
//	int		handle		handle for this channel
//	VPB_DIGITS	*newdig		get digits parameters
//	char		*buf		destination buffer
{
	VPB_DIGITS_INT	*d;
	int				ret, terminate;
	USHORT			digit;
	int				retcode = -1;

	try {
		ValidHandleCheck(handle);

		// dont want MMQ adding digits at same time we are reading
		// them out.

		GenericEnterCriticalSection(&DigitsSect);

		// copy params to local storage

		d = &digits[handle];		
		
		validate_digits(newdig->term_digits);
		strcpy(d->term_digits,newdig->term_digits);
		if (newdig->max_digits > MAX_DIGITS)
			throw Wobbly(VPBAPI_DIGITS_MAX_DIGITS_TOO_BIG);
		
		d->max_digits = newdig->max_digits;
		d->digit_time_out = newdig->digit_time_out;
		d->inter_digit_time_out = newdig->inter_digit_time_out;
		d->so_far = 0;
		d->state = ACTIVE;
		d->buf = buf;
		d->digit_timer->timer_start();
		d->inter_digit_timer->timer_stop();	// timer in reset
		d->async = 0;
		GenericLeaveCriticalSection(&DigitsSect);

		// now keep polling for terminate conditions

		do {
			ret = digbuf_read(dbuf, handle, &digit);
		
			// wop it into buffer 
			
			terminate = 0;
			if (ret == OK) {
				d->inter_digit_timer->timer_start();
				d->buf[d->so_far] = (char)digit;
				(d->so_far)++;
			
				terminate = (d->so_far == d->max_digits) 
					      || digit_match((char)digit, d->term_digits);
			}
			else
				GenericSleep(SLEEPMS);

			// if dbuf has enough digits terminate

			if (d->state && terminate && d->so_far == d->max_digits) {
				d->buf[d->so_far] = 0;
				retcode = VPB_DIGIT_MAX;
				d->state = INACTIVE;
			}

			// if terminate digit found terminate

			if (d->state && terminate && digit_match((char)digit, d->term_digits)) {
				d->buf[d->so_far] = 0;
				retcode = VPB_DIGIT_TERM;
				d->state = INACTIVE;
			}

			// check digit time out

			USHORT time_out;
			d->digit_timer->timer_check_time_out_ms(d->digit_time_out, &time_out);
			if (d->state && (time_out == TIME_OUT)) {
				d->buf[d->so_far] = 0;
				retcode = VPB_DIGIT_TIME_OUT;
				terminate = 1;
				d->state = INACTIVE;
			}

			// check inter digit time out

			d->inter_digit_timer->timer_check_time_out_ms(d->inter_digit_time_out, &time_out);
			if (d->state && (!terminate) && (time_out == TIME_OUT)) {
				d->buf[d->so_far] = 0;
				retcode = VPB_DIGIT_INTER_DIGIT_TIME_OUT;
				terminate = 1;
				d->state = INACTIVE;
			}

		} while(!terminate);
	
		d->inter_digit_timer->timer_stop();	// timer in reset
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_get_digits_sync"));
	}
	
	assert(retcode != -1);
	return(retcode);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_flush_digits
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Clears the internal digit buffer, often called before 
	vpb_get_digits_xxxx.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_flush_digits(int handle)
{
	try {
		ValidHandleCheck(handle);
		GenericEnterCriticalSection(&DigitsSect);
		digbuf_clear(dbuf, handle);
		digbuf_clear(dbufrec, handle);
		GenericLeaveCriticalSection(&DigitsSect);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_flush_digits"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: digits_new_digit
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Called by the MMQ when DSP detects a digits event.  Places the digit in
	the internal digit buffer, then adds this digit to the user buffer
	if active.

\*--------------------------------------------------------------------------*/

void digits_new_digit(int h, USHORT digit) {
	VPB_DIGITS_INT	*d;
	
	// make sure main thread doesnt interfere with d while we are
	// pissing about with it

	GenericEnterCriticalSection(&DigitsSect);
	d = &digits[h];		
	digbuf_write(dbuf, h, digit);
	if(d->record){
		digbuf_write(dbufrec, h, digit);
		if ((d->state == ACTIVE) && d->async)
			check_buffer_record(h);
	}
	else{
		if ((d->state == ACTIVE) && d->async && (!d->voxtron))
			check_buffer(h);
		if ((d->state == ACTIVE) && d->async && d->voxtron)
			check_buffer_voxtron(h);
	}
	GenericLeaveCriticalSection(&DigitsSect);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: digit_match
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Determines if the digit is a member of the term_digit string.

\*--------------------------------------------------------------------------*/

static int digit_match(char digit, char *term_digits)
{
	char *p = term_digits;

	while(*p != 0) {
		if (*p == digit)
			return(1);
		p++;
	}

	return(0);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: validate_digits
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Determines if the digit string contains valid characters and is of
	valid length.

\*--------------------------------------------------------------------------*/

static void	validate_digits(char *digits)
{
	int i,j;
	
	// validate this entries digit string

	USHORT len = strlen(digits);
	if (len > MAX_TERM)
		throw Wobbly(VPBAPI_DIGITS_TERM_DIGITS_STRING_TOO_LONG);

	for(j=0; j<len; j++) {
		// search for digit in table

		char c = toupper(digits[j]);
		int num = -1;
		for(i=0; i<NUM_VALID_DIG; i++) {
			if (valid_dig[i] == c)
				num = i;
		}
		if (num < 0)
			throw Wobbly(VPBAPI_DIGITS_INVALID_TERM_DIGITS);
	}

}

/*---------------------------------------------------------------------------*\

	FUNCTION: check_buffer
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Reads digits from dbuf into the user buffer and determines if the
	terminate conditions have been met.  If so a terminate event is
	posted.
\*--------------------------------------------------------------------------*/

void check_buffer(int handle) {
	VPB_DIGITS_INT	*d;
	int				ret, terminate;
	USHORT			digit;
	VPB_EVENT		e;
	
	GenericEnterCriticalSection(&DigitsSect);
	d = &digits[handle];		

	assert(d->state == ACTIVE);
	assert(d->async);
	assert(!d->voxtron);
	assert(!d->record);
	
	do {
		ret = digbuf_read(dbuf, handle, &digit);
		
		// wop it into buffer 
				
		terminate = 0;
		if (ret == OK) {
			d->inter_digit_timer->timer_start();
			d->buf[d->so_far] = (char)digit;
			(d->so_far)++;
					
			terminate = (d->so_far == d->max_digits) 
					  || digit_match((char)digit, d->term_digits);
		}

		// if dbuf has enough digits terminate

		if (d->state && terminate && d->so_far == d->max_digits) {
			d->buf[d->so_far] = 0;
			e.type = VPB_DIGIT;
			e.handle = handle;
			e.data = VPB_DIGIT_MAX;
			e.data1 = d->so_far;
			putevt(&e, VPB_MDIGIT);
			d->state = INACTIVE;
		}

		// if terminate digit found terminate

		if (d->state && terminate && digit_match((char)digit, d->term_digits)) {
			d->buf[d->so_far] = 0;
			e.type = VPB_DIGIT;
			e.handle = handle;
			e.data = VPB_DIGIT_TERM;
			e.data1 = d->so_far;
			putevt(&e, VPB_MDIGIT);
			d->state = INACTIVE;
		}

	} while((ret == OK) && (!terminate));

	
	GenericLeaveCriticalSection(&DigitsSect);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: check_buffer_voxtron
	AUTHOR..: David Rowe & John Kostogiannis
	DATE....: 15/4/99

	Reads digits from dbuf into the user buffer and determines if the
	terminate conditions have been met.  If so a terminate event is
	posted.
	Similar to check_buffer except that it can also terminate if 
	the destination buffer is full.

\*--------------------------------------------------------------------------*/

void check_buffer_voxtron(int handle) {
	VPB_DIGITS_INT	*d;
	int				ret, terminate;
	USHORT			digit;
	VPB_EVENT		e;
	
	GenericEnterCriticalSection(&DigitsSect);
	d = &digits[handle];		

	assert(d->state == ACTIVE);
	assert(d->async);
	assert(d->voxtron); 
//	assert(!d->record);
	
	do {
		ret = digbuf_read(dbuf, handle, &digit);
			
		// wop it into buffer 
				
		terminate = 0;
		if (ret == OK) {
			d->inter_digit_timer->timer_start();
			d->buf[d->so_far] = (char)digit;
			(d->so_far)++;
				
			terminate = (d->so_far == d->max_digits) 
					  || digit_match((char)digit, d->term_digits) 
					  ||(d->so_far >=(d->size-1));
		}

		// if destination buffer is full terminate

		if (d->state && terminate && (d->so_far >= d->size-1)) {
			d->buf[d->so_far] = 0;
			e.type = VPB_DIGIT;
			e.handle = handle;
			e.data = VPB_DIGIT_BUFFER_FULL;
			e.data1 = d->so_far;
			putevt(&e, VPB_MDIGIT);
			d->state = INACTIVE;
		}

		// if dbuf has enough digits terminate
		if (d->state && terminate && d->so_far == d->max_digits) {
			d->buf[d->so_far] = 0;
			e.type = VPB_DIGIT;
			e.handle = handle;
			e.data = VPB_DIGIT_MAX;
			e.data1 = d->so_far;
			putevt(&e, VPB_MDIGIT);
			d->state = INACTIVE;
		}

		// if terminate digit found terminate

		if (d->state && terminate && digit_match((char)digit, d->term_digits)) {
			d->buf[d->so_far] = 0;
			e.type = VPB_DIGIT;
			e.handle = handle;
			e.data = VPB_DIGIT_TERM;
			e.data1 = d->so_far;
			putevt(&e, VPB_MDIGIT);
			d->state = INACTIVE;
		}

	} while((ret == OK) && (!terminate));
	
	GenericLeaveCriticalSection(&DigitsSect);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: check_buffer_record
	AUTHOR..: John Kostogiannis
	DATE....: 28/4/99

	Reads digits from dbufrec into the user buffer and determines if the
	terminate conditions have been met.  If so a terminate event is
	posted. The termination conditions for this are maxdigits and term_digits
	and time_out.
\*--------------------------------------------------------------------------*/

void check_buffer_record(int handle) {
	VPB_DIGITS_INT	*d;
	int				ret, terminate;
	USHORT			digit;
	VPB_EVENT		e;
	
	GenericEnterCriticalSection(&DigitsSect);
	d = &digits[handle];		
	assert(d->state == ACTIVE);
	assert(d->async);
	assert(d->record);
	do {
		ret = digbuf_read(dbufrec, handle, &digit);
			
		// wop it into buffer 
				
		terminate = 0;
		if (ret == OK) {
			d->buf[d->so_far] = (char)digit;
			(d->so_far)++;
				
			terminate = (d->so_far == d->max_digits) 
					  || digit_match((char)digit, d->term_digits);
		}

		// if dbuf has enough digits terminate

		if (d->state && terminate && d->so_far == d->max_digits) {
			d->buf[d->so_far] = 0;
			e.type = VPB_DIGIT;
			e.handle = handle;
			e.data = VPB_DIGIT_MAX;
			e.data1 = d->so_far;
			putevt(&e, VPB_MDIGIT);
			d->state = INACTIVE;
			d->record = 0;
		}

		// if terminate digit found terminate

		if (d->state && terminate && digit_match((char)digit, d->term_digits)) {
			d->buf[d->so_far] = 0;
			e.type = VPB_DIGIT;
			e.handle = handle;
			e.data = VPB_DIGIT_TERM;
			e.data1 = d->so_far;
			putevt(&e, VPB_MDIGIT);
			d->state = INACTIVE;
			d->record = 0;
		}

	} while((ret == OK) && (!terminate));
	
	GenericLeaveCriticalSection(&DigitsSect);
}


/*---------------------------------------------------------------------------*\

	FUNCTION: check_timers
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Called periodically from MMQ to determine if any time outs have 
	occured while processing the async version of get digits.

\*--------------------------------------------------------------------------*/

void digits_check_timers() {
	VPB_DIGITS_INT	*d;
	int				handle;
	VPB_EVENT		e;
	USHORT time_out;
	
	GenericEnterCriticalSection(&DigitsSect);

	for(handle=0; handle<NumCh; handle++) {
		d = &digits[handle];		
		
		// for obtaining digits async while recording only have a time out
		// no interdigittimeout

		if(!d->record){
			if ((d->state == ACTIVE) && d->async) {
				// check digit time out

				d->digit_timer->timer_check_time_out_ms(d->digit_time_out, &time_out);
				if (d->state && (time_out == TIME_OUT)) {
					d->buf[d->so_far] = 0;
					e.data = VPB_DIGIT_TIME_OUT;
					e.type = VPB_DIGIT;
					e.data1 = d->so_far;
					e.handle = handle;
					putevt(&e, VPB_MDIGIT);
					d->state = INACTIVE;
					d->voxtron = 0;
				}

				// check inter digit time out

				d->inter_digit_timer->timer_check_time_out_ms(d->inter_digit_time_out, &time_out);
				if (d->state && (time_out == TIME_OUT)) {
					d->buf[d->so_far] = 0;
					e.data = VPB_DIGIT_INTER_DIGIT_TIME_OUT;
					e.type = VPB_DIGIT;
					e.data1 = d->so_far;
					e.handle = handle;
					putevt(&e, VPB_MDIGIT);
					d->state = INACTIVE;
					d->voxtron = 0;
				}
			}
		}
		else{
		   if ((d->state == ACTIVE) && d->async) {
				// check digit time out

				d->digit_timer->timer_check_time_out_ms(d->digit_time_out, &time_out);
				if (d->state && (time_out == TIME_OUT)) {
					d->buf[d->so_far] = 0;
					e.data = VPB_DIGIT_TIME_OUT;
					e.type = VPB_DIGIT;
					e.data1 = d->so_far;
					e.handle = handle;
					putevt(&e, VPB_MDIGIT);
					d->state = INACTIVE;
					d->record = 0;
				}
		   }
		}
	}

	GenericLeaveCriticalSection(&DigitsSect);
}


/*---------------------------------------------------------------------------*\

	FUNCTION: digits_term
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Translates the terminate code to a string for debug purposes.

\*--------------------------------------------------------------------------*/

char *digits_term(int data)
{
	if ((data > 4) || (data < 0))
		return term_str[5];
	else
		return(term_str[data]);
}


